/**************************************
	DML_timer.h
	
	introduces the timer class, for simplification of timers.
	only supports up to 1000 Hz, although QNX will do 2000 Hz.
	
	
	author:  ryan findley
	
	history:
	when	who		what
	-----------------------------------
	5/11/99	ryan	created
	7/29/99 ryan	added checkProxyBackup,
					operator pid_t(),
	8/4/99	ryan	added clearProxyQueue
	8/11/99 ryan	added FREQ definitions, recognized 1000 Hz is limit for this class
****************************************/
#ifndef DML_TIMER_H
#define DML_TIMER_H


//-------------------- public includes --------------------
#include <time.h>
#include <sys/proxy.h>
#include <sys/types.h>
#include <sys/kernel.h>
#include <signal.h>

//-------------------- public definitions -----------------
const char 	ONE_SHOT = 'o';
const int 	PROXY_DNE  = -1;
const pid_t SELF = 0;

const int FREQ1Hz = 1000;
const int FREQ2Hz = 500;
const int FREQ3Hz = 333;
const int FREQ4Hz = 250;
const int FREQ5Hz = 200;
const int FREQ6Hz = 166;
const int FREQ7Hz = 143;
const int FREQ8Hz = 125;
const int FREQ9Hz = 111;
const int FREQ10Hz = 100;
const int FREQ15Hz = 66;
const int FREQ20Hz = 50;
const int FREQ40Hz = 25;
const int FREQ50Hz = 20;
const int FREQ100Hz = 10;
const int FREQ200Hz = 5;
const int FREQ250Hz = 4;
const int FREQ500Hz = 2;
const int FREQ1000Hz = 1;

inline int timerFreqHz( const int hz )
{
	return( 1000/hz );
}

//-------------------- class definitions ------------------

class DML_timer
{
  public:
    timer_t id;
    pid_t proxy;
    pid_t proxyTarget;
    int proxyPriority;
        
    DML_timer();						//creates a blank timer instance... not armed
    DML_timer(const DML_timer &);		//copy constructor... does NOT create a new timer, just copies data associated with it.
    DML_timer(int, pid_t = SELF);		//creates a timer with period of (period) ms
    DML_timer(int, char, pid_t = SELF);	//sets up a one-shot timer
    ~DML_timer();						//destructor

	inline wait() { Receive(proxy, 0, 0); }	
    setPeriod(int);						//sets the timer period
    arm();                              //arms the timer: starts it going.
    stop();                             //temporarily stop the timer from ticking
    restart();                          //restart a stopped timer
    timeLeft(struct itimerspec &);      //reports the time left
    int checkProxyBackup();             //checks how many proxies are backed up. returns PROXY_DNE if error occurs
	clearProxyQueue();					//clears the proxy queue by destroying the proxy and re-making the timer.
										//this is kind of slow, and will screw up any other function who has a copy
										//of the old proxy pid in memory, so use this function with care.
    operator pid_t();                   //returns the proxy

  private:
    struct itimerspec timing;
    struct sigevent event;

};


inline DML_timer::operator pid_t()
{
	return(proxy);
}


#endif //DML_TIMER_H
