/*************************************************

child.cpp

introduces the child class. Provides functions for handling
child processes.

Author: Ryan Findley


History:
who	when		what
--------------------------------
ryan	5/10/99		started

***************************************************/

#define NOT_TEST

//-------------------- include files --------------------------
#include <string.h>	//C strings
#include <signal.h>
#include <stdio.h>
#include <sys/sched.h>
#include <sys/kernel.h>
#include <sys/mman.h>

#include "DML_child.hpp"
//-------------------- private definitions --------------------
#define MAX_NAME_LENGTH 31

//-------------------- private prototypes ---------------------


//-------------------- module code ----------------------------

/************************************************************
	child()
	default constructor
	
	Creates a blank child instance, to be filled later.

	child(char *name)
	useful constructor

	Creates a child process, the executable "name".
*************************************************************/
DML_child::DML_child()	
{
    id = -1;
    proxy = -1;
    priority = -1;
    verbosity = LOUD;
    flags = P_NOWAIT;
    strcpy(name, "");
	nChildren++;
	theChildren[nChildren-1] = this;
}

//init the static members

int DML_child::nChildren = 0;
DML_child ** DML_child::theChildren = new (DML_child *)[MAX_N_CHILD]


static DML_child::killChildren()
{
	delete[] theChildren;	
}

DML_child::DML_child(char * theName, int theVerbosity)
{
    id = -1;
    proxy = -1;
    priority = -1;
    flags = P_NOWAIT;
	verbosity = theVerbosity;
	if(verbosity == LOUD)
	{
		printf("*** creating child: %s... ",theName);
		fflush(stdout);
	}
    open(theName);
    attachProxy();	//just 'cause it's useful
    sendInfoMsg();
	if(id != -1 && verbosity == LOUD)
		printf("...success!\n");
	else if(verbosity == LOUD)
		printf("...failure!\n");
}
    
DML_child::~DML_child()
{
    if(verbosity == LOUD)
	printf("closing child: %s.  \n",name);
    kill(id,SIGTERM);
    id = -1;
    strcpy(name,"");
}

/****************************************************************
	open(char * theName)
	opens up the process (spawns)

	sendInfoMsg()
	sends the default info message, containing info.

	int sendMsg(int theMsg)
	sends an integer message and returns the integer response.
	it's so common, I figured I'd include it as a member.
*****************************************************************/
DML_child::open(char * theName)
{
    pid_t theChildPid;
    
    theChildPid = spawnlp(flags,theName,theName,NULL);
    if (theChildPid == -1 && verbosity == LOUD)
    {
    	printf("New process error: %s  \n",name);
    	id = -1;
    }
	
	strncpy(name,theName,MAX_NAME_LENGTH);
	if(strlen(theName) > MAX_NAME_LENGTH && verbosity == LOUD)
		printf("The child name selected, %s, was too long.\n Truncated to %s.\n",theName,name);
    id = theChildPid;
}

DML_child::sendInfoMsg()
{
    int status;
    msg.proxy = proxy;
    msg.verbosity = verbosity;
        
    status = Send(id, &msg, &rmsg, sizeof(childInfoStruct), sizeof(childInfoStruct) );
    if (status == -1 && verbosity == LOUD) //process doesn't exist, or interrupted with a signal
    {
    	printf("error sending message to child: %s  \n",name);
    }
    priority = rmsg.priority;
}

int DML_child::sendMsg(int theMsg)	//returns -1 on error: -1 can still be valid receive data, so watch out
{
    int theRmsg;
    int status;
    
    status = Send(id, &theMsg, &theRmsg, sizeof(int), sizeof(int) );
    if (status == -1 && verbosity == LOUD) //process doesn't exist, or interrupted with a signal
    {
    	printf("error sending message to child: %s  \n",name);
    	return(-1);
    }
    return(theRmsg);
}

/******************************************************************
	int beQuiet() and int beLoud()
	changes the verbosity, returns the previous verbosity.
*******************************************************************/
int DML_child::beQuiet()
{
    int oldPrio;
    oldPrio = verbosity;
    verbosity = QUIET;
    return(oldPrio);
}

int DML_child::beLoud()
{
    int oldPrio;
    oldPrio = verbosity;
    verbosity = LOUD;
    return(oldPrio);
}

/******************************************************************
	attachProxy();
	attaches a proxy to the child. the InfoMsg must be sent AFTER attaching a proxy.
	if failure in any way, proxy will be -1
	
	kick();
	triggers the child's proxy. The proxy must exist. returns -1 on error, 0 on success
*******************************************************************/
DML_child::attachProxy()
{
    pid_t theProxy;
    proxy = qnx_proxy_attach(id,0,0,-1);
    if (proxy == -1 && verbosity == LOUD)
       	printf("Error attaching proxy to child:  %s  \n",name);
}    
    
int DML_child::kick()
{
    int status;
    
    if (id != -1)
	status = Trigger(proxy);
    else
	return(-1);
	
    return(status);
}


#ifdef TEST
//test harness

#include <conio.h>
void endMaster(int sigCount);

DML_child child("child");

void main(void)
{
	
	signal(SIGTERM,endMaster);

	fflush(stdin);
	getch();
	printf("\nstuff from master:\n");
	printf("master PID: %d\n",getpid());
	printf("verbosity: %d\n",LOUD);
	printf("proxy:  %d\n",child.proxy);
	printf("priority: %d\n",child.priority);
	printf("child PID: %d\n",child.id);
	
}	

void endMaster(int sigCount)
{
    child.~DML_child();
}



#endif //TEST